// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "degub.h"
#include "hardware.h"
#include "hw_internal.h"
#include "gp.h"
#include "array.h"
#include "messages.h"
#include "main.h"
#include "gp_internal.h"
#include "gp_texture.h"
#include "hw_gx.h"
#include "hw_vi.h"
#include "gp_dlrec.h"
#include <algorithm>
#include <d3dx9tex.h>
#include <process.h>

void GP::activate_gp() {
	if(!hw->m_gp_activated) {
		hw->m_gp_activated = true;
		HWGLE(PostMessage(hw->hWnd, LM_GP_ACTIVATED, 0, 0));
	}
}

//we could take the MemInterface from Hardware...
GP::GP(MemInterface &_m, LPDIRECT3DDEVICE9 pd3dDevice, Hardware *h)
: mem(_m), hw(h) {
	ZERO_OBJECT(vs_key);
	ZERO_OBJECT(vs_control);
	//ZERO_OBJECT(vs_data);
	ZERO_OBJECT(ps_key);
	ZERO_OBJECT(ps_control);
	ZERO_OBJECT(ps_current);
	ZERO_OBJECT(stat);
	ZERO_OBJECT(tlut);
	ZERO_ARRAY(tx);
	ZERO_OBJECT(m);

	m.bp_mask = 0xffffff;
	m.cullmode = D3DCULL_NONE;

	/*//directx defaults
	m.bp_reg[0x40] = 0x000017;
	m.xf_mem[0x100E] = 0x00000001;*/

#define SET_FUNCTION(array, id, function) array[id] = function
	//opcodes
	void (GP::*fp)() = &GP::unknown_opcode;
	memset_dword(m_arr_opcode, MAKE(DWORD, fp), 0x100);

#define SET_OPCODE(opcode, function) SET_FUNCTION(m_arr_opcode, opcode, &GP::function);\
	SET_FUNCTION(m_rec_opcode, opcode, &GP::dlr_##function)

	SET_OPCODE(0x00, nop);
	SET_OPCODE(0x08, load_cp);
	SET_OPCODE(0x10, load_xf);
	SET_OPCODE(0x20, load_xf_indx_a);
	SET_OPCODE(0x28, load_xf_indx_b);
	SET_OPCODE(0x30, load_xf_indx_c);
	SET_OPCODE(0x38, load_xf_indx_d);
	SET_OPCODE(0x40, call_list);
	SET_OPCODE(0x44, update_metrics);
	SET_OPCODE(0x48, invalidate_vertex_cache);
	SET_OPCODE(0x61, load_bp);
	for(BYTE vat=0; vat<8; vat++) {
		SET_OPCODE(0x80+vat, draw_quads);
		SET_OPCODE(0x90+vat, draw_triangles);
		SET_OPCODE(0x98+vat, draw_triangle_strip);
		SET_OPCODE(0xA0+vat, draw_triangle_fan);
		SET_OPCODE(0xA8+vat, draw_lines);
		SET_OPCODE(0xB0+vat, draw_line_strip);
		SET_OPCODE(0xB8+vat, draw_points);
	}
	//SET_OPCODE(0x0E, strange_opcode);
	//SET_OPCODE(0xFF, strange_opcode);

	//BP registers
	void (GP::*bp)(DWORD) = &GP::bp_direct;
	memset_dword(m_arr_bp, MAKE(DWORD, bp), 0x100);

#define SET_BP(reg, function) SET_FUNCTION(m_arr_bp, reg, &GP::function)

	SET_BP(0x00, bp_gen_mode);
	SET_BP(0x20, bp_su_scis0);
	SET_BP(0x21, bp_su_scis1);
	SET_BP(0x22, bp_su_lpsize);
	for(BYTE i=0; i<8; i++) {
		SET_BP(0x28+i, bp_tev_order);
	}
	SET_BP(0x40, bp_pe_zmode);
	SET_BP(0x41, bp_pe_cmode0);
	SET_BP(0x45, bp_pe_done);
	SET_BP(0x47, bp_pe_token);
	SET_BP(0x48, bp_pe_token_int);
	SET_BP(0x49, bp_efb_topleft);
	SET_BP(0x4A, bp_efb_bottomright);
	SET_BP(0x52, bp_pe_copy_execute);
	SET_BP(0x59, bp_scissor_offset);
	SET_BP(0x64, bp_tlut_load0);
	SET_BP(0x65, bp_tlut_load1);
	SET_BP(0x66, bp_tx_invalidate);
	for(BYTE i=0; i<4; i++) {
		SET_BP(0x80+i, bp_tx_setmode0);
		SET_BP(0xA0+i, bp_tx_setmode0);
		SET_BP(0x84+i, bp_tx_setmode1);
		SET_BP(0xA4+i, bp_tx_setmode1);
		SET_BP(0x88+i, bp_tx_setimage0);
		SET_BP(0xA8+i, bp_tx_setimage0);
		SET_BP(0x8C+i, bp_tx_setimage1);
		SET_BP(0xAC+i, bp_tx_setimage1);
		SET_BP(0x90+i, bp_tx_setimage2);
		SET_BP(0xB0+i, bp_tx_setimage2);
		SET_BP(0x94+i, bp_tx_setimage3);
		SET_BP(0xB4+i, bp_tx_setimage3);
		SET_BP(0x98+i, bp_tx_settlut);
		SET_BP(0xB8+i, bp_tx_settlut);
	}
	for(BYTE i=0; i<16; i++) {
		SET_BP(0xC0+i*2, bp_tev_color);
		SET_BP(0xC1+i*2, bp_tev_alpha);
	}
	for(BYTE i=0; i<4; i++) {
		SET_BP(0xE0+i*2, bp_tev_register_l);
		SET_BP(0xE1+i*2, bp_tev_register_h);
	}
	SET_BP(0xF3, bp_tev_alphafunc);
	for(BYTE i=0; i<=8; i++) {
		SET_BP(0xF6+i, bp_tev_ksel);
	}
	SET_BP(0xFE, bp_mask);

	//Direct3D
	if(pd3dDevice == NULL) {
		throw hardware_fatal_exception("pD3DDevice is NULL!");
	}
	m.pd3dDevice = pd3dDevice;

	//thread
	HWGLE(m.hEvent = CreateEvent(NULL, false, false, NULL));
	HWGLE(m.hThread = (HANDLE)_beginthreadex(NULL, 0, ThreadFunc, this, 0, NULL));
	//HWGLE(SetThreadPriority(m.hThread, THREAD_PRIORITY_ABOVE_NORMAL));
}

bool GP::endThread() {
	VGPDEGUB("Setting GP Quit signal\n");
	hw->fifo.gp.quit = true;
	TGLE(SetEvent(m.hEvent));
	TGLE(MyResumeThread(m.hThread));
	VGPDEGUB("GP Quit signal set\n");
	DWORD res = WaitForSingleObject(m.hThread, 4000);  //arbitrary timeout
	TGLECUSTOM(WAIT_FAILED, res);
	if(res == WAIT_TIMEOUT) {
		//TGLE(TerminateThread(m.hThread, 0));
		FAIL(UE_GPTHREAD_TIMEOUT);
	}
	TGLE(GetExitCodeThread(m.hThread, &res));
	if(res == GPTR_ERROR) {
		DEGUB("GP Error: %s\n", errorstring.c_str());
	}
	if(g::verbose) {
		MYFILETIME creation, exit, kernel, user;
		TGLE(GetThreadTimes(m.hThread, &creation.ft, &exit.ft, &kernel.ft, &user.ft));
		DEGUB("GP Thread:\n");
		dumpFileTime("Lifetime", exit.qword - creation.qword);
		if(m.ftActivate.qword == 0) {
			DEGUB("Thread was never activated.\n");
		} else {
			dumpFileTime("Active time", exit.qword - m.ftActivate.qword);
		}
		dumpFileTime("Kernel time", kernel.qword);
		dumpFileTime("User time", user.qword);
		DEGUB("\n");
	}
	return true;
}

GP::~GP() {
	GLE(endThread());
	GLE(CloseHandle(m.hThread));
	GLE(CloseHandle(m.hEvent));
	HR(endScene());
	HR(setClearState());

	//clear the shader caches
	for(VSHashMap::iterator itr = vs_map.begin(); itr != vs_map.end(); itr++) {
		SAFE_RELEASE(itr->second.pVertexDeclaration);
		SAFE_RELEASE(itr->second.pVertexShader);
	}
	for(PSHashMap::iterator itr = ps_map.begin(); itr != ps_map.end(); itr++) {
		SAFE_RELEASE(itr->second.pPixelShader);
	}

	if(m_ctx_map.size() > 0) {
		DEGUB("%i textures cached\n", m_ctx_map.size());
		CTX_HASHMAP::iterator itr = m_ctx_map.begin();
		while(itr != m_ctx_map.end()) {
			SAFE_RELEASE(itr->second.p);
			GPDEGUB("0x%08X (%i iterations)\n", itr->first, itr->second.nchanges);
			itr++;
		}
	}

	if(hw->m_gp_activated) {
		stat.dump();
	}
}

void GP::STATS::dump() {
	DEGUB("GP Statistics:\n");
	DEGUB("Total bytes: %I64i\n", bytes_ndl + bytes_dl);
	__int64 odraw = odraw_quads + odraw_trilist + odraw_tristrip +
		odraw_trifan + odraw_linelist + odraw_linestrip +
		odraw_points + odraw_null;
	__int64 tops = obp + ocp + oxf + oxfa + oxfb + oxfc + oxfd + odl + odraw + nops + ivcs;
	DEGUB("Total draw ops: %I64i\n", odraw);
	__int64 triangles = quads * 2 + trislist + trisstrip + trisfan;
	__int64 lines = lineslist + linesstrip;
	DEGUB("Total vertices: %I64i\n", triangles*3 + lines*2 + points);
	DEGUB("Total triangles (including quads): %I64i\n", triangles);
	DEGUB("Total lines: %I64i\n", lines);
	DEGUB("Finished ops: %I64i\n", tops);

#define GPSTAT_DUMP_I64(id, desc) DEGUB("%s: %I64i\n", desc, id);
	GP_STATISTICS(GPSTAT_DUMP_I64);
	DEGUB("\n");
	if(tops != ops && tops != ops - 1) {	//ops may be one larger if an op was interrupted
		DEGUB("GP Stat discrepancy!\n");
	}
}

void GP::getVerboseText(ostream& str) {
	static __int64 pFrames=0, pVS=0, pPS=0;
	str << "VS "<< stat.vertexshaders <<"\nPS "<< stat.pixelshaders <<"\n";
	str << "VS/f "<< ((stat.vertexshaders - pVS) / MAX(stat.frames - pFrames, 1)) <<"\n"
		"PS/f "<< ((stat.pixelshaders - pPS) / MAX(stat.frames - pFrames, 1)) <<"\n";
	pFrames = stat.frames;
	pVS = stat.vertexshaders;
	pPS = stat.pixelshaders;
}

void dump_matrix(const D3DMATRIX *m) {
	if(m == NULL) {
		GPDEGUB("NULL pointer!\n");
	} else {
		GPDEGUB("%.6g %.6g %.6g %.6g\n%.6g %.6g %.6g %.6g\n%.6g %.6g %.6g %.6g\n%.6g %.6g %.6g %.6g\n",
			m->_11, m->_12, m->_13, m->_14,
			m->_21, m->_22, m->_23, m->_24,
			m->_31, m->_32, m->_33, m->_34,
			m->_41, m->_42, m->_43, m->_44);
	}
}

HRESULT GP::beginScene() {
	if(!m.in_scene) {
		GPDEGUB("BeginScene\n");
		THR(m.pd3dDevice->BeginScene());
		m.in_scene = true;

		GPHR(setRS(D3DRS_FILLMODE, g::gp_wireframe ? D3DFILL_WIREFRAME : D3DFILL_SOLID));
		GPHR(setRS(D3DRS_CULLMODE, g::gp_wireframe ? D3DCULL_NONE : m.cullmode));
		GPHR(setRS(D3DRS_SHADEMODE, g::gp_flat ? D3DSHADE_FLAT : D3DSHADE_GOURAUD));
	}
	return S_OK;
}
HRESULT GP::endScene() {
	GPDEGUB("EndScene\n");
	if(m.in_scene) {
		THR(m.pd3dDevice->EndScene());
		m.in_scene = false;
	}
	return S_OK;
}

HRESULT GP::setSS(DWORD Sampler, D3DSAMPLERSTATETYPE Type, DWORD Value) {
	DWORD oldValue;
	THR(m.pd3dDevice->GetSamplerState(Sampler, Type, &oldValue));
	if(oldValue != Value) {
		THR(m.pd3dDevice->SetSamplerState(Sampler, Type, Value));
	}
	return S_OK;
}
HRESULT GP::setRS(D3DRENDERSTATETYPE State, DWORD Value) {
	DWORD oldValue;
	THR(m.pd3dDevice->GetRenderState(State, &oldValue));
	if(oldValue != Value) {
		THR(m.pd3dDevice->SetRenderState(State, Value));
	}
	return S_OK;
}
HRESULT GP::setTSS(DWORD Stage, D3DTEXTURESTAGESTATETYPE Type, DWORD Value) {
	DWORD oldValue;
	THR(m.pd3dDevice->GetTextureStageState(Stage, Type, &oldValue));
	if(oldValue != Value) {
		THR(m.pd3dDevice->SetTextureStageState(Stage, Type, Value));
	}
	return S_OK;
}

inline DWORD fifoGetStopPt(DWORD readptr, DWORD writeptr, DWORD breakpt, bool bp_enabled)
{
	if(bp_enabled) {
		if(writeptr >= readptr)
			if(breakpt >= readptr)
				return MIN(writeptr, breakpt);
			else
				return writeptr;
		else
			if(breakpt >= readptr)
				return breakpt;
			else
				return MIN(writeptr, breakpt);
	} else {
		return writeptr;
	}
}
void GP::updateStopPt() {
	m.fifoStopPt = fifoGetStopPt(m.fifo.readptr, m.fifo.writeptr, m.fifo.breakpoint,
		m.fifo.bp_enabled);
}
void GP::fifo_set() {
	if(m.in_list)
		throw hardware_fatal_exception("Internal error: GP FIFO set in Display List!\n");
	FIFODEGUB("GP FIFO set!\n");
	m.fifo = MAKE(GP_FIFO, hw->fifo.gp);
	m.alignedFifoEnd = PAD_ENDPTR(m.fifo.end);
	m.fifoSize = m.alignedFifoEnd - m.fifo.base;
	updateStopPt();
	if(g::verbose && g::gp_log) {
		DUMPDWORD(m.alignedFifoEnd);
		DUMPDWORD(m.fifoSize);
		DUMPDWORD(m.fifoStopPt);
	}
	m.pFifo = mem.getp_physical(m.fifo.base, m.fifoSize);
}
const BYTE *GP::queue_get_data(size_t size) {
	const int MAX_SIZE = 32;  //must not be > 32
	static BYTE temp[MAX_SIZE];
	MYASSERT(size <= MAX_SIZE);
	FIFODEGUB("GP Writeptr: 0x%08X\n", m.fifo.writeptr);
	FIFODEGUB("GP Readptr: 0x%08X + 0x%X = ", m.fifo.readptr, size);
	DWORD newreadptr = m.fifo.readptr + size;
	DWORD oldreadptr = m.fifo.readptr;
	while((newreadptr >= m.alignedFifoEnd) ?
		(newreadptr - m.fifoSize > m.fifoStopPt)
		: (m.fifo.readptr <= m.fifoStopPt && newreadptr > m.fifoStopPt))
	{
		if(m.in_list) {
			DUMPDWORD(newreadptr);
			DUMPDWORD(m.alignedFifoEnd);
			DUMPDWORD(newreadptr - m.fifoSize);
			DUMPDWORD(m.fifoStopPt);
			throw hardware_fatal_exception("GP Display List ended in the middle of an opcode!");
		}
		bool do_wait = true;
		if(m.fifo.bp_enabled && m.fifoStopPt == m.fifo.breakpoint) {
			GPDEGUB("GP Breakpoint reached!\n");
			hw->fifo.gp.idle = true;
			hw->fifo.gp.read_enabled = false;
			hw->fifo.gp.breakpoint_reached = true;
			hw->hwh(0x0000, hw->hrh(0x0000) | 0x10);
			hw->interrupt.raise(INTEX_CP, "GP Breakpoint");
		} else { //hw->fifo.gp.writeptr may have changed
			DWORD writeptr = hw->fifo.gp.writeptr;
			if(writeptr != m.fifo.writeptr) {
				m.fifo.writeptr = writeptr;
				updateStopPt();
				do_wait = false;
			}
		}
		if(do_wait) {
			hw->fifo.gp.readptr = FLOOR_32B(m.fifo.readptr);
			HwGX::do_underflow(hw);
			waitForMoreData();
			//hw->fifo.gp may have changed
			DWORD temp = hw->fifo.gp.readptr;
			if(temp == FLOOR_32B(oldreadptr)) {
				newreadptr = oldreadptr + size;
			} else {
				//if(m.fifo.in_command)
				//throw hardware_fatal_exception("GP FIFO readptr change while in command!");
				oldreadptr = temp;
				newreadptr = temp + size;
			}
			m.fifo = MAKE(GP_FIFO, hw->fifo.gp);
			updateStopPt();
		}
		FIFODEGUB("GP Writeptr: 0x%08X\n", m.fifo.writeptr);
		FIFODEGUB("GP StopPt: 0x%08X\n", m.fifoStopPt);
		FIFODEGUB("GP Readptr: 0x%08X + 0x%X = ", oldreadptr, size);
	}
	DWORD base = m.fifo.base;
	m.fifo.readptr = newreadptr;
	(m.in_list ? stat.bytes_dl : stat.bytes_ndl) += size;
	if(m.fifo.readptr >= m.alignedFifoEnd) {
		if(!m.in_list)
			m.fifo.readptr -= m.fifoSize;
		FIFODEGUB("0x%08X\n", m.fifo.readptr);
		DWORD j=0;
		for(DWORD i=oldreadptr - base; i<m.fifoSize; i++) {
			temp[j++] = m.pFifo[i];
		}
		if(!m.in_list) for(DWORD i=0; i<(m.fifo.readptr - base); i++) {
			temp[j++] = m.pFifo[i];
		}
		FIFODEGUB("GP wrapRead %i: 0x", size);
		for(size_t i=0; i<size; i++) {
			FIFODEGUB("%02X", temp[i]);
		}
		FIFODEGUB("\n");
		MYASSERT(j == size && j < sizeof(temp));
		return temp;
	} else {
		FIFODEGUB("0x%08X\n", m.fifo.readptr);
		FIFODEGUB("GP read %i: 0x", size);
		for(size_t i=0; i<size; i++) {
			FIFODEGUB("%02X", (m.pFifo + (oldreadptr - base))[i]);
		}
		FIFODEGUB("\n");
		return m.pFifo + (oldreadptr - base);
	}
}
BYTE GP::queue_get_past_byte(size_t npos) {
	DWORD ptr = m.fifo.readptr - npos;
	if(ptr < m.fifo.base)
		ptr += m.fifoSize;
	return m.pFifo[ptr - m.fifo.base];
}
DWORD GP::queue_get_24bits() {
	return make_24bits((BYTE*)queue_get_data(3));
}
void GP::waitForMoreData() {
	hw->fifo.gp.idle = true;
	hw->fifo.gp.in_command = m.fifo.in_command;
	FIFODEGUB("GP Waiting for more data...\n");
	do {
		HWGLERCUSTOM(WAIT_OBJECT_0, WaitForSingleObject(m.hEvent, INFINITE));
		FIFODEGUB("GP Event raised\n");
		if(hw->fifo.gp.quit) {
			FIFODEGUB("GP Quit signal recieved\n");
			throw emulation_stop_exception("GP Quit");
		}
		if(!hw->fifo.gp.read_enabled) {
			FIFODEGUB("GP Read still disabled\n");
		}
	} while(!hw->fifo.gp.read_enabled);
	hw->fifo.gp.idle = false;
}
void GP::execute_opcode() {
	BYTE opcode = GP_QUEUE_GET_BYTE;
	stat.ops++;
	m.fifo.in_command = true;
	(this->*m_arr_opcode[opcode])();
	m.fifo.in_command = false;
}

unsigned __stdcall GP::ThreadFunc(void *arg) {
	GP *gp = (GP *)arg;
	try {
		gp->waitForMoreData();	//fifo is not initialized yet
		gp->activate_gp();
		GetSystemTimeAsFileTime(&gp->m.ftActivate.ft);
loop:
		gp->execute_opcode();
		goto loop;
	} catch(hardware_fatal_exception &e) {
		return gp->handle_exception("hardware_fatal_exception", e.what());
	} catch(generic_fatal_exception &e) {
		return gp->handle_exception("generic_fatal_exception", e.what());
	} catch(page_fault_exception &e) {
		return gp->handle_exception("page_fault_exception", e.what());
	} catch(rec_fatal_exception &e) {
		return gp->handle_exception("rec_fatal_exception", e.what());
	} catch(bad_form_exception &e) {
		return gp->handle_exception("bad_form_exception", e.what());
	} catch(interp_fatal_exception &e) {
		return gp->handle_exception("interp_fatal_exception", e.what());
	} catch(bouehr_exception &e) {
		return gp->handle_exception("bouehr_exception", e.what());
	} catch(ui_exception &e) {
		return gp->handle_exception("ui_exception", e.what());
	} catch(emulation_stop_exception &e) {
		DEGUB("Caught emulation_stop_exception in GP thread: %s\n", e.what());
		return GPTR_SUCCESS;
	} catch(std::exception &e) {
		return gp->handle_exception("exception", e.what());
	}
}
unsigned GP::handle_exception(const char *type, const char *error) {
	g_capp.SuspendEmuThread();
	DEGUB("Caught %s in GP thread: %s\n", type, error);
	errorstring = error;
	hw->fifo.gp.idle = true;
	GLE(PostMessage(hw->hWnd, LM_GPEXCEPTION, 0, 0));
	return GPTR_ERROR;
}

void GP::unknown_opcode() {
	BYTE opcode = queue_get_past_byte(1);
	DEGUB("GP Unknown opcode: 0x%02X\n", opcode);
	throw hardware_fatal_exception("GP Unknown opcode!");
}
void GP::nop() {
	GPDEGUB("GP NOP\n");
	stat.nops++;
}
void GP::invalidate_vertex_cache() {
	GPDEGUB("GP Invalidate Vertex Cache\n");
	stat.ivcs++;
}
void GP::update_metrics() {
	throw hardware_fatal_exception("GP Update Metrics unemulated!");
}

void GP::call_list() {
	DWORD address = GP_QUEUE_GET_DWORD;
	DWORD size = GP_QUEUE_GET_DWORD;
	GPDEGUB("\nGP Call List 0x%08X, 0x%X\n", address, size);
	if(!IS_32B_ALIGNED(address) || !IS_32B_ALIGNED(size))
		throw hardware_fatal_exception("GP Call List unaligned!");
	if(m.in_list)
		throw hardware_fatal_exception("GP Display List tried to call another list!");
	stat.odl++;
	m.in_list = true;

	DWORD old_fifoStopPt = m.fifoStopPt;
	DWORD old_alignedFifoEnd = m.alignedFifoEnd;
	DWORD old_fifoSize = m.fifoSize;
	const BYTE *old_pFifo = m.pFifo;
	GP_FIFO old_fifo = m.fifo;
	m.pFifo = mem.getp_translated(address, size);
	m.fifoSize = size;
	m.alignedFifoEnd = PHYSICALIZE(address) + size;
	m.fifoStopPt = m.alignedFifoEnd;
	static GP_FIFO dlfifo;
	ZERO_OBJECT(dlfifo);
	//these are used
	dlfifo.base = PHYSICALIZE(address);
	dlfifo.writeptr = m.alignedFifoEnd;
	dlfifo.readptr = dlfifo.base;
	//these aren't
	dlfifo.dist = size;
	dlfifo.end = m.alignedFifoEnd - 4;

	m.fifo = dlfifo;
	if(g::gp_dlrec) {
		runCompiledList();
	} else {
		do {
			execute_opcode();
			RELEASE_ASSERT(m.fifo.base == PHYSICALIZE(address));
			RELEASE_ASSERT(m.fifo.writeptr == m.alignedFifoEnd);
			RELEASE_ASSERT(m.fifo.dist == size);
			RELEASE_ASSERT(m.fifo.end == m.alignedFifoEnd - 4);
		} while(m.fifo.readptr < m.fifo.writeptr);
	}
	m.fifo = old_fifo;
	m.pFifo = old_pFifo;
	m.fifoSize = old_fifoSize;
	m.alignedFifoEnd = old_alignedFifoEnd;
	m.fifoStopPt = old_fifoStopPt;

	m.in_list = false;
	GPDEGUB("End of list\n\n");
}

void GP::load_cp() {
	BYTE reg = GP_QUEUE_GET_BYTE;
	DWORD data = GP_QUEUE_GET_DWORD;
	cpload(reg, data);
}

void GP::cpload(BYTE reg, DWORD data) {
	m.cp_reg[reg] = data;
	GPDEGUB("GP CP Load 0x%02X, 0x%08X\n", reg, data);
	stat.ocp++;
}

HRESULT GP::checkTextureRequirements(UINT width, UINT height, D3DFORMAT format,
																		 DWORD usage)
{
#define DTTT_ARGS(macro) macro(UINT, width, width) macro(UINT, height, height)\
	macro(UINT, mipmap, 1) macro(D3DFORMAT, format, format)
#define DTTT_DECLARE(type, name, defval) type c##name = defval;
	DTTT_ARGS(DTTT_DECLARE);

	HRESULT res = D3DXCheckTextureRequirements(m.pd3dDevice, &cwidth, &cheight,
		&cmipmap, usage, &cformat, D3DPOOL_DEFAULT);

	bool error = false;
#define DTTT_TEST(type, name, defval) if(c##name != defval) error = true;
	DTTT_ARGS(DTTT_TEST);

	if(error) {
		DEGUB("Texture doesn't fit!\n");
#define DTTT_DUMP(type, name, defval) DEGUB("%s: %i -> %i\n", #name, defval, c##name);
		DTTT_ARGS(DTTT_DUMP);
	}
#undef DTTT_ARGS
#undef DTTT_DECLARE
#undef DTTT_TEST
#undef DTTT_DUMP

	if(res != D3DERR_NOTAVAILABLE && res != D3D_OK) {
		return res;
	}
	if(res == D3DERR_NOTAVAILABLE)
		throw hardware_fatal_exception("Your graphics card has insufficient texture caps.");
	MYASSERT(res == D3D_OK);

	if(usage == D3DUSAGE_RENDERTARGET) {
		LPDIRECT3D9 pD3D=NULL;
		HWHR(m.pd3dDevice->GetDirect3D(&pD3D));
		res = pD3D->CheckDeviceFormatConversion(D3DADAPTER_DEFAULT,
			D3DDEVTYPE_HAL, D3DFMT_X8R8G8B8, cformat);
		SAFE_RELEASE(pD3D);
		if(res != D3DERR_NOTAVAILABLE && res != D3D_OK) {
			return res;
		}
		if(res == D3DERR_NOTAVAILABLE)
			throw hardware_fatal_exception("Your graphics card has insufficient conversion caps.");
		MYASSERT(res == D3D_OK);
	}

	return res;
}

/*bool GP::isMatch(LPDIRECT3DTEXTURE9 pTex, UINT width, UINT height, D3DFORMAT format,
DWORD usage)
{
#define DTTT_ARGS(macro) macro(UINT, Width, width) macro(UINT, Height, height)\
macro(DWORD, Usage, usage) macro(D3DFORMAT, Format, format)
#define DTTT_DECLARE(type, name, defval) type c##name = defval;
DTTT_ARGS(DTTT_DECLARE);

HRESULT res = D3DXCheckTextureRequirements(m.pd3dDevice, &cWidth, &cHeight,
NULL, usage, &cFormat, D3DPOOL_DEFAULT);
if(res != D3DERR_NOTAVAILABLE && res != D3D_OK) {
HWHR(res);
}
D3DSURFACE_DESC desc;
GPHR(pTex->GetLevelDesc(0, &desc));

bool error = false;
#define DTTT_TEST(type, name, defval) if(c##name != desc.name) error = true;
DTTT_ARGS(DTTT_TEST);
#undef DTTT_ARGS
#undef DTTT_DECLARE
#undef DTTT_TEST

return error;
}*/

void GP::createLoadTexture(LPDIRECT3DTEXTURE9 *ppTex, UINT width, UINT height,
													 D3DFORMAT format, UINT pitch, const void *src)
{
	GPHR(checkTextureRequirements(width, height, format, 0));
	GPHR(D3DXCreateTexture(m.pd3dDevice, width, height, 1, 0, format, D3DPOOL_DEFAULT,
		ppTex));
	RECT rect = { 0, 0, width, height };
	LPDIRECT3DSURFACE9 pSurface=NULL;
	GPHR((*ppTex)->GetSurfaceLevel(0, &pSurface));
	GPHR(D3DXLoadSurfaceFromMemory(pSurface, NULL, NULL, src, format,
		pitch, NULL, &rect, D3DX_FILTER_NONE, 0));
	SAFE_RELEASE(pSurface);
}

/*void GP::do_cache_line_create(DWORD address, DWORD line) {
tcache.lines[line] = address;
}
void GP::do_cache_line_select(DWORD address, DWORD line) {
if(tcache.lines[line] == address)
return;
else if(tcache.lines[line] == 0)
tcache.lines[line] = address;
else
throw hardware_fatal_exception("GP Tex cache error no.5");
}*/

void GP::do_texture(BYTE index) {
	MYASSERT(tx[index].changed);
	tx[index].changed = false;
	if(!tx[index].address)
		return;
	VGPDEGUB("Loading texture %i @ 0x%08X\n", index, tx[index].address);
	CTX_HASHMAP::iterator itr = m_ctx_map.find(tx[index].address);
	//bool created = false;
	if(itr == m_ctx_map.end()) {
		if(mem.prw(tx[index].address) == tx[index].address)
			throw hardware_fatal_exception("GP Tex cache error no.6");
		CTEXTURE texture;
		texture.nchanges = 0;
		texture.p = createTexture(tx[index].address, tx[index].width, tx[index].height,
			tx[index].format, tx[index].tlutformat, tx[index].tlutoffset, 0);
		itr = m_ctx_map.insert(CTX_PAIR(tx[index].address, texture)).first;
		mem.pww(tx[index].address, tx[index].address);
		GPDEGUB("Texture 0x%08X created\n", tx[index].address);
		//created = true;
	} else {
		if(mem.prw(tx[index].address) == tx[index].address) {
			GPDEGUB("Texture 0x%08X selected\n", tx[index].address);
		} else {
			//we may want to look out for partial format changes and cache invalidates, too
			GPDEGUB("Texture 0x%08X changed\n", tx[index].address);
			itr->second.p->Release();
			itr->second.p = createTexture(tx[index].address, tx[index].width, tx[index].height,
				tx[index].format, tx[index].tlutformat, tx[index].tlutoffset,
				itr->second.nchanges);
			mem.pww(tx[index].address, tx[index].address);
			itr->second.nchanges++;
		}
	}

	/*if(!tx.preloaded) {
	DWORD baseline = tx.even_offset >> 15;
	for(DWORD i=baseline; i<baseline + tx.even_size; i++) {
	if(created)
	do_cache_line_create(tx.address, i);
	else
	do_cache_line_select(tx.address, i);
	}
	baseline = tx.odd_offset >> 15;
	for(DWORD i=baseline; i<baseline + tx.odd_size; i++) {
	if(created)
	do_cache_line_create(tx.address, i);
	else
	do_cache_line_select(tx.address, i);
	}
	}*/

	GPHR(m.pd3dDevice->SetTexture(index, g::gp_wireframe ? NULL : itr->second.p));
}
